﻿using System;
using System.Runtime.InteropServices;
using System.Text;
using System.Windows.Forms;

namespace PWS.API.Socket.Server
{
    public partial class FormServer : Form
    {
        #region API Defination

        [StructLayout(LayoutKind.Sequential)]
        public struct WSAData
        {
            const int WSADescriptionLength = 256;
            const int WSASystemStatusLength = 128;

            public Int16 Version;
            public Int16 HighVersion;
            [MarshalAs(UnmanagedType.ByValTStr, SizeConst = WSADescriptionLength + 1)]
            public String Description;
            [MarshalAs(UnmanagedType.ByValTStr, SizeConst = WSASystemStatusLength + 1)]
            public String SystemStatus;
            public Int16 MaxSockets;
            public Int16 MaxUdpDg;
            public IntPtr VendorInfo;
        }

        public const short Success = 0;
        public const short HighVersion = 2;
        public const short LowVesion = 2;
        public const short WordVersion = HighVersion * 256 + LowVesion;

        [DllImport("ws2_32.dll")]
        public static extern Int32 WSAStartup(short wVersionRequested, ref WSAData wsaData);

        [DllImport("ws2_32.dll")]
        public static extern Int32 WSACleanup();

        public enum SocketType : short
        {
            SocketStream = 0x1,
            SocketDatagram = 0x2,
            SocketRaw = 0x3,
            SocketRDM = 0x4,
            SocketSEQPacketStream = 0x5
        }

        public enum AddressFamilies : short
        {
            AF_UNSPEC = 0x0,
            AF_UNIX = 0x1,
            AF_INET = 0x2,
            AF_IMPLINK = 0x3,
            AF_PUP = 0x4,
            AF_CHAOS = 0x5,
            AF_NS = 0x6,
            AF_IPX = 0x6,
            AF_ISO = 0x7,
            AF_OSI = 0x7,
            AF_ECMA = 0x8,
            AF_DATAKIT = 0x9,
            AF_CCITT = 0xA,
            AF_SNA = 0xB,
            AF_DECnet = 0xC,
            AF_DLI = 0xD,
            AF_LAT = 0xE,
            AF_HYLINK = 0xF,
            AF_APPLETALK = 0x10,
            AF_NETBIOS = 0x11,
            AF_VOICEVIEW = 0x12,
            AF_FIREFOX = 0x13,
            AF_UNKNOWN1 = 0x14,
            AF_BAN = 0x15,
            AF_ATM = 0x16,
            AF_INET6 = 0x17,
            AF_CLUSTER = 0x18,
            AF_12844 = 0x19,
            AF_IRDA = 0x1A,
            AF_NETDES = 0x1C,
            AF_TCNPROCESS = 0x1D,
            AF_TCNMESSAGE = 0x1E,
            AF_ICLFXBM = 0x1F
        }

        public enum ProtocolType : short
        {
            IP = 0x0,
            ICMP = 0x1,
            IGMP = 0x2,
            GGP = 0x3,
            TCP = 0x6,
            PUP = 0xC,
            UDP = 0x11,
            IDP = 0x16,
            IPV6 = 0x29,
            ND = 0x4D,
            ICLFXBM = 0x4E,
            Raw = 0xFF,
            Max = 0x100
        }

        [DllImport("ws2_32.dll")]
        public static extern IntPtr socket(AddressFamilies addressFamily, SocketType socketType, ProtocolType protocolType);

        [StructLayout(LayoutKind.Sequential)]
        public struct SocketAddress
        {
            public AddressFamilies AddressFamily;
            public ushort Port;
            public int IP;
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = 8)]
            public byte[] sin_zero;
        }

        [DllImport("Ws2_32.dll")]
        public static extern int bind(IntPtr socketHandle, ref SocketAddress address, int addressSize);

        [DllImport("Ws2_32.dll")]
        public static extern int listen(IntPtr socketHandle, int backlog);

        [DllImport("Ws2_32.dll")]
        public static extern ushort htons(ushort hostShort);

        [DllImport("Ws2_32.dll")]
        public static extern uint inet_addr(string ipString);

        [DllImport("Ws2_32.dll")]
        public static extern IntPtr accept(IntPtr socketHandle, ref  SocketAddress socketAddress, ref int socketAddressSize);

        [Flags]
        public enum MessageFlags : int
        {
            Default = 0x0,

            /// <summary>
            /// Processes Out Of Band (OOB) data.
            /// </summary>
            OutOfBand = 0x1,

            /// <summary>
            /// Peeks at the incoming data. The data is copied into the buffer,
            /// but is not removed from the input queue.
            /// </summary>
            Peek = 0x2,

            /// <summary>
            /// send without using routing tables
            /// </summary>
            DoNotRoute = 0x4,

            /// <summary>
            /// The receive request will complete only when one of the following events occurs:
            /// </summary>
            WaitAll = 0x8,

            /// <summary>
            /// partial send or recv for message xport
            /// </summary>
            Partial = 0x8000,

            /// <summary>
            /// ???? ... ???
            /// </summary>
            DoNotWait = 0x1000000
        }

        [DllImport("Ws2_32.dll")]
        public static extern int send(IntPtr socketHandle, byte[] buffer, int length, MessageFlags flags);

        [DllImport("Ws2_32.dll")]
        public static extern int recv(IntPtr socketHandle, byte[] buffer, int length, MessageFlags flags);

        [DllImport("ws2_32.dll")]
        public static extern int closesocket(IntPtr socketHandle);

        #endregion API Defination

        private IntPtr serverSocket;
        private IntPtr clientSocket;
        const int BufferSize = 1024;
        private byte[] buffer = new byte[BufferSize];

        public FormServer()
        {
            InitializeComponent();
        }

        private void ButtonListenClick(object sender, EventArgs e)
        {
            WSAData data = new WSAData();

            int result = WSAStartup(WordVersion, ref data);
            if (result != Success)
            {
                MessageBox.Show(@"خطا در بارگذاری تنظیمات پیش فرض");
                return;
            }
            serverSocket = socket(AddressFamilies.AF_INET, SocketType.SocketStream, ProtocolType.TCP);
            ushort port;
            if (!ushort.TryParse(textBoxPort.Text, out port))
            {
                MessageBox.Show(@"لطفا برای شماره پورت یک عدد صحیح از 1024 تا 65535 مشخص نمایید");
                return;
            }
            SocketAddress socketInfo = new SocketAddress();
            socketInfo.AddressFamily = AddressFamilies.AF_INET;
            socketInfo.IP = (int)inet_addr("127.0.0.1"); //(int)htonl(INADDR_ANY);
            socketInfo.Port = htons(port);
            bind(serverSocket, ref socketInfo, Marshal.SizeOf(socketInfo));
            result = listen(serverSocket, 0);
            if (result == -1)
            {
                MessageBox.Show(@"خطا در عملیات");
                return;
            }
            //button1_Click(null, null);
            timerWaitForRequest.Start();
            buttonDisconnect.Enabled = true;
            buttonListen.Enabled = false;
            textBoxMessage.Enabled = true;
        }

        private void TextBoxPortTextChanged(object sender, EventArgs e)
        {
            buttonListen.Enabled = textBoxPort.Text.Trim() != String.Empty;
        }

        private void ButtonDisconnectClick(object sender, EventArgs e)
        {
            if (serverSocket == IntPtr.Zero || serverSocket.ToInt32() == -1) return;
            int result = closesocket(serverSocket);
            if (result == -1)
            {
                MessageBox.Show(@"خطا در قطع ارتباط");
                return;
            }
            textBoxPort.Text = String.Empty;
            buttonListen.Enabled = false;
            buttonDisconnect.Enabled = false;
            buttonSend.Enabled = false;
            textBoxMessage.Enabled = false;
        }

        private void ButtonCloseClick(object sender, EventArgs e)
        {
            Application.Exit();
        }

        private void TimerWaitForRequestTick(object sender, EventArgs e)
        {
            if (serverSocket == IntPtr.Zero) return;
            SocketAddress socketInfo = new SocketAddress();
            int socketSize = Marshal.SizeOf(socketInfo);  //marshal andaze socket info mizare too socket saize
            clientSocket = accept(serverSocket, ref socketInfo, ref socketSize);
            timerRecive.Start();
            timerWaitForRequest.Stop();
        }

        private void TimerReciveTick(object sender, EventArgs e)
        {
            byte[] buffer = new byte[BufferSize];
            int reciveBytes = recv(clientSocket, buffer, BufferSize, 0);
            if (reciveBytes <= 0) return;
            string message = Encoding.UTF8.GetString(buffer);
            textBoxLog.Text += Environment.NewLine + String.Format("He > {0} ", message);
            timerRecive.Stop();
        }

        private void ButtonSendClick(object sender, EventArgs e)
        {
            try
            {
                if (clientSocket == IntPtr.Zero || clientSocket.ToInt32() == -1 || textBoxMessage.Text.Trim() == String.Empty) return;
                string data = textBoxMessage.Text;
                buffer = Encoding.UTF8.GetBytes(data);
                int sendBytes = send(clientSocket, buffer, buffer.Length, 0);

                if (sendBytes > 0)
                {
                    textBoxLog.Text += Environment.NewLine + String.Format("You > {0}", data);
                    textBoxMessage.Text = string.Empty;
                    timerRecive.Start();
                }
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.ToString());
            }
        }

        private void TextBoxMessageTextChanged(object sender, EventArgs e)
        {
            buttonSend.Enabled = textBoxMessage.Text.Trim() != String.Empty;
        }

        private void ButtonClearClick(object sender, EventArgs e)
        {
            textBoxLog.Text = string.Empty;
        }
    }
}